import asyncio
import os
import sys
import time
import traceback


from pylog.pylogger import PyLogger

from py_pli.pylib import VUnits, GlobalVar, Measurements
from predefined_tasks.common.helper import send_to_gc, create_report_header_str, report_path, \
    mini_initialize_instrument, send_error_message_to_gc
from virtualunits.HAL import HAL
from virtualunits.vu_node_application import VUNodeApplication

import config_enum.scan_table_enum as scan_table_enum
import config_enum.excitationlight_selector_enum as els_enum
import config_enum.filter_module_slider_enum as fms_enum
import config_enum.bottom_light_director_enum as bld_enum
import config_enum.focus_mover_enum as fm_enum
import config_enum.platedoor_enum as pd_enum
import config_enum.detector_aperture_slider_enum as das_enum
import config_enum.stacker_enum as stacker_enum

PATH, SCRIPT = os.path.split(__file__)
BASE_NAME = SCRIPT.split('.')[0]

# Logging and Messages
logger = PyLogger.logger
error_msg = f"Error Predefined Tasks in {BASE_NAME}."
start_msg = f"Start Predefined Task {BASE_NAME}."
info_msg = f"Information from {BASE_NAME}."
stop_msg = f"Stopped Predefined Task {BASE_NAME}."
complete_msg = f"Completed Predefined Task {BASE_NAME}."

log_prefix = "gc_predefined_tasks> "

# Boards

hal_unit: HAL = VUnits.instance.hal
eef_unit: VUNodeApplication = hal_unit.nodes['EEFNode']
fmb_unit: VUNodeApplication = hal_unit.nodes['Mainboard']
mc6_unit: VUNodeApplication = hal_unit.nodes['MC6']

# mover-dict in order of homing

all_movers = {
    'fm': hal_unit.focusMover,
    'st': hal_unit.scan_table,
    'pd': hal_unit.plateDoor,
    'fms': hal_unit.filterModuleSlider,
    'das1': hal_unit.detectorApertureSlider1,
    'das2': hal_unit.detectorApertureSlider2,
    'els': hal_unit.excitationLightSelector,
    'bld': hal_unit.bottomLightDirector,
    'sd': hal_unit.doorLatch,
    'usfm': hal_unit.usLumFocusMover}


if VUnits.instance.hal.stackerLeft and VUnits.instance.hal.stackerRight:
    add_stacker_vu = {'st_left_lift': VUnits.instance.hal.stackerLeft.lift_mover,
                      'st_left_release': VUnits.instance.hal.stackerLeft.release_mover,
                      'st_right_lift': VUnits.instance.hal.stackerRight.lift_mover,
                      'st_right_release': VUnits.instance.hal.stackerRight.release_mover}
    all_movers.update(add_stacker_vu)

# High Level Function to start in GC

async def pd_stutter_test(profile = 9, cycles = 1, min_step = 0.5):
    # Stuttertest Plate Door
    # Do a stutter movement of the mover wih the test profile #9
    # - Disconnect Power MC6 Board
    # - Connect Mover to M and ref - connector
    # - Connect Power MC6 Board
    # (- Press Restart in GC)
    # - Start Script

    mover = 'pd'
    test_name = sys._getframe().f_code.co_name    # get function name from sys

    t_mover = all_movers[mover]
    start = t_mover.get_config(pd_enum.Positions.Min) + min_step
    stop = t_mover.get_config(pd_enum.Positions.Max) - min_step
    result = await mover_stutter_test(mover, profile, start, stop, min_step, cycles)
    msg = f'{test_name} Result: {result}'
    
    return(msg)

async def das1_stutter_test(profile = 9, cycles = 1, min_step = 0.5):
    # Stuttertest
    # Do a stutter movement of the mover wih the test profile #9
    # - Disconnect Power MC6 Board
    # - Connect Mover to M and ref - connector
    # - Connect Power MC6 Board
    # (- Press Restart in GC)
    # - Start Script

    mover = 'das1'
    test_name = sys._getframe().f_code.co_name    # get function name from sys

    t_mover = all_movers[mover]
    start = t_mover.get_config(das_enum.Positions.Min) + min_step
    stop = t_mover.get_config(das_enum.Positions.Max) - min_step
    result = await mover_stutter_test(mover, profile, start, stop, min_step, cycles)
    msg = f'{test_name} Result: {result}'
    return(msg)

async def das2_stutter_test(profile = 9, cycles = 1, min_step = 0.5):
    # Stuttertest
    # Do a stutter movement of the mover wih the test profile #9
    # - Disconnect Power MC6 Board
    # - Connect Mover to M and ref - connector
    # - Connect Power MC6 Board
    # (- Press Restart in GC)
    # - Start Script

    mover = 'das2'
    test_name = sys._getframe().f_code.co_name    # get function name from sys   
    
    t_mover = all_movers[mover]
    start = t_mover.get_config(das_enum.Positions.Min) + min_step
    stop = t_mover.get_config(das_enum.Positions.Max) - min_step
    result = await mover_stutter_test(mover, profile, start, stop, min_step, cycles)
    msg = f'{test_name} Result: {result}'
    return(msg)

async def els_stutter_test(profile = 9, cycles = 1, min_step = 0.1):
    # Stuttertest
    # - Disconnect Power MC6 Board
    # - Connect Mover to M and ref - connector
    # - Connect Power MC6 Board
    # (- Press Restart in GC)
    # - Start Script

    mover = 'els'
    test_name = sys._getframe().f_code.co_name    # get function name from sys  
    
    start = 0.0
    stop = 359.0
    result = await mover_stutter_test(mover, profile, start, stop, min_step, cycles)
    msg = f'{test_name} Result: {result}'
    return(msg)

# Basic Functions

async def module_mover_position_test(mover='pd', profile=9, num_of_cycles=10):
    # Copy of /predefined_tasks/mover_tests.mover_position_test
    # checks saved positions
    # Mover:
    # fm: FocusMover
    # fms: FilterModuleSlider
    # pd: PlateDoor
    # das1: DetectorApertureSlider1
    # das2: DetectorApertureSlider2
    # els: ExcitationLightSelector: Producing high step errors end_pos > 355 (>5000msteps)
    # bld: BottomLightDirector: Producing high step errors end_pos > 345 (>5000msteps)

    test_name = mover + '_' + sys._getframe().f_code.co_name    # get function name from sys

    t_mover = all_movers[mover]
    
    # FLE-2167 Module Test Setup Changes
    await eef_unit.StartFirmware() 
    await t_mover.InitializeDevice()


    cycles = int(num_of_cycles) if (num_of_cycles != "") else 1
    max_step_error = 0
    step_error_sum = 0

    msg = f'Starting {test_name}...'
    logger.info(log_prefix + msg)
    await send_to_gc(msg)
    

    report_file = report_path(test_name)

    with open(report_file, 'w') as report:

        header = create_report_header_str()
        report.write(header)

        if t_mover:
            try:
                GlobalVar.set_stop_gc(False)
                # FLE-2167 Module Test Setup Changes
                # await mini_initialize_instrument()
                # if mover != 'pd':
                #    await plate_door.Open()

                if mover == 'fm':
                    saved_positions = t_mover.configManager.get_section(fm_enum.Positions)
                elif mover == 'pd':
                    saved_positions = t_mover.configManager.get_section(pd_enum.Positions)
                elif mover == 'das1':
                    saved_positions = t_mover.configManager.get_section(das_enum.Positions)
                elif mover == 'das2':
                    saved_positions = t_mover.configManager.get_section(das_enum.Positions)
                elif mover == 'els':
                    saved_positions = t_mover.configManager.get_section(els_enum.Positions)
                elif mover == 'bld':
                    saved_positions = t_mover.configManager.get_section(bld_enum.Positions)
                elif mover == 'usfm':
                    saved_positions = t_mover.configManager.get_section(fm_enum.Positions)
                elif mover == 'fms':
                    saved_positions = t_mover.configManager.get_section(fms_enum.Positions).copy()
                    filter_offset = t_mover.get_config(fms_enum.GC_Params.IndexOffset_c)
                    filter1_pos = t_mover.get_config(fms_enum.Positions.IndexZeroPosMeas)
                    filter1_bcr_pos = t_mover.get_config(fms_enum.Positions.IndexZeroPosBCR_c)
                    for i in range(1, 6):
                        filter_i_pos = filter1_pos + i*filter_offset
                        filter_i_bcr_pos = filter1_bcr_pos + i*filter_offset
                        saved_positions.update({f"filter{i+1}_pos" : filter_i_pos})
                        saved_positions.update({f"filter{i+1}_bcr_pos": filter_i_bcr_pos})
                elif mover == 'st_left_lift':
                    saved_positions = t_mover.configManager.get_section(stacker_enum.Positions_Lift)
                elif mover == 'st_left_release':
                    saved_positions = t_mover.configManager.get_section(stacker_enum.Positions_Release)
                elif mover == 'st_right_lift':
                    saved_positions = t_mover.configManager.get_section(stacker_enum.Positions_Lift)
                elif mover == 'st_right_release':
                    saved_positions = t_mover.configManager.get_section(stacker_enum.Positions_Release)

                if not saved_positions:
                    msg = f"{test_name} : No saved position was found in the mover configuration file"
                    logger.error(log_prefix + msg)
                    await send_to_gc(msg)
                    report.write(msg + "\n")
                    GlobalVar.set_stop_gc(True)
                    return

                await t_mover.Home()
                await t_mover.UseProfile(profile)

                for i in range(cycles):
                    for name, position in saved_positions.items():
                        if GlobalVar.get_stop_gc():
                            msg = f"{test_name} : stopped by User"
                            logger.error(log_prefix + msg)
                            await send_to_gc(msg)
                            return

                        msg = f"Cycle {i+1}: Moving {mover} to Position {name} => {position}"
                        logger.info(log_prefix + msg)
                        await send_to_gc(msg)
                        report.write(msg + "\n")
                        await t_mover.Move(position)
                        await asyncio.sleep(0.5)

                    await t_mover.Home()
                    step_error = t_mover.Mover.LastHomeStepErrors
                    step_error_sum += step_error
                    if abs(step_error) > abs(max_step_error):
                        max_step_error = step_error
                    msg = f"Cycle {i+1} [step error]: {step_error}"
                    logger.info(log_prefix + msg)
                    report.write(msg + "\n")
                    await send_to_gc(msg)

                await t_mover.Home()
                msg = f"{test_name} step error sum = {step_error_sum}"
                logger.info(log_prefix + msg)
                report.write(msg + "\n")
                await send_to_gc(msg)
                msg = f"{test_name}_result [max step error]: {max_step_error}"
                logger.info(log_prefix + msg)
                report.write(msg + "\n")
                await send_to_gc(msg)

                logger.info("gc_predefined_tasks> Execution completed.")
                # FLE-2167 Module Test Setup Changes
                # return
                return(step_error)

            except Exception as ex:
                msg = f"{test_name} -> {traceback.format_exc()}"
                logger.error(log_prefix + msg)
                report.write(msg + "\n")
                await send_error_message_to_gc(msg, ex)

                GlobalVar.set_stop_gc(True)  # Stop script execution
                return

        else:
            msg = f"{test_name} cannot be started. {mover} is not available."
            logger.info(log_prefix + msg)
            report.write(msg + "\n")
            await send_to_gc(msg)

async def module_scan_table_test(cycles: int = 1, profile: int = 9, plate_format: str = "384", meas_position: str = "fbdtop", delay_ms: int = 10):
    """
    params:
        cycles: (int) number of test cycles
        profile: (int) profile id
        plate_format: (int) allowed formats are 96, 384, 1536
        meas_position: (str) detector where the scan table movements are performed below/above
                    allowed measurement positions are "us", "fbdtop", "fbdbottom".
        delay_ms: (int) this is the simulated mesaurement time in a well in milliseconds.

    description:
        The scan table iterates over the selected plate format below the selected detector "cycles" times
        with selected profile and returns step errors of each cycle, sum of all step errors and maximum step errors.
    """

    test_name = sys._getframe().f_code.co_name  # get function name from sys

    try:
        GlobalVar.set_stop_gc(False)
        scan_table = all_movers["st"]
        
        # FLE-2167 Module Test Setup Changes
        await scan_table.InitializeDevice()


        format = str(plate_format) if str(plate_format) in ["96", "384", "1536"] else "384"
        mover_profile = int(profile) if (profile != "") else 3
        nr_of_cycles = int(cycles) if (cycles != "") else 1
        detector = meas_position if meas_position in ["fbtop", "fbdbottom", "us"] else "fbdtop"
        delay = int(delay_ms)/1000 if (delay_ms != "") else 10/1000

        max_step_errors_x = 0
        max_step_errors_y = 0
        step_error_sum_x = 0
        step_error_sum_y = 0

        if format == "384":
            plate_type = "'384 OptiPlate (Black)'"
            rows = 16
            cols = 24
        if format == "96":
            plate_type = "'96 OptiPlate (Black)'"
            rows = 8
            cols = 12
        if format == "1536":
            plate_type = "'1536 OptiPlate (Black)'"
            rows = 32
            cols = 48

        # set detector meas pos
        if detector == 'us':
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.USLum_TopLeftCorner)
        elif detector == 'fbdtop':
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.FBDTop_Flash1_TopLeftCorner)
        elif detector == 'fbdbottom':
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.FBDBottom_Flash1_TopLeftCorner)

        scan_table.SetPlateType(plate_type)

        report_file = report_path(test_name)

        with open(report_file, 'w') as report:

            header = create_report_header_str()
            report.write(header)

            #await scan_table.Home()
            await mini_initialize_instrument()
            await plate_door.Open()
            await scan_table.UseProfile(1)
            await scan_table.MoveToWell(1, 1)
            await scan_table.UseProfile(mover_profile)

            msg = f"Starting {test_name} ..."
            logger.info(log_prefix + msg)
            await send_to_gc(msg)

            for i in range(nr_of_cycles):
                start_row = time.time()
                for row in range(1, rows + 1, 1):
                    if GlobalVar.get_stop_gc():
                        msg = f"{test_name} : stopped by User"
                        logger.error(log_prefix + msg)
                        await send_to_gc(msg)
                        return
                    start_col = time.time()
                    if row % 2 == 1:
                        for col in range(1, cols + 1):
                            await scan_table.MoveToWell(col, row)
                            time.sleep(delay)
                            msg = f'cycle {i}: well x, y, {col}, {row}'
                            logger.info(log_prefix + msg)
                            await send_to_gc(msg)
                            report.write(msg + "\n")
                    else:
                        for col in range(cols, 0, -1):
                            await scan_table.MoveToWell(col, row)
                            time.sleep(delay)
                            msg = f'cycle {i}: well x, y, {col}, {row}'
                            logger.info(log_prefix + msg)
                            await send_to_gc(msg)
                            report.write(msg + "\n")
                    t_col = (time.time() - start_col) / cols
                t_row = (time.time() - start_row) / rows
                step_errors = await scan_table.Home()
                step_error_x = step_errors[0]
                if len(step_errors) > 1:
                    step_error_y = step_errors[1]
                else:
                    step_error_y = 0
                step_error_sum_x += step_error_x
                step_error_sum_y += step_error_y
                if abs(step_error_x) > abs(max_step_errors_x):
                    max_step_errors_x = step_error_x
                if abs(step_error_y) > abs(max_step_errors_y):
                    max_step_errors_y = step_error_y
                msg = f'{test_name} cycle {i+1} : step errors = {step_errors}, time per well = {t_col:.3f}s, time per row = {t_row:.3f}s'
                await send_to_gc(msg)
                report.write(msg + "\n")

            msg = f'{test_name} : step error sum = {step_error_sum_x}; {step_error_sum_y}'
            await send_to_gc(msg)
            report.write(msg + "\n")

            msg = f"{test_name}_result [max step error]: {max_step_errors_x}; {max_step_errors_y}"
            logger.info(log_prefix + msg)
            report.write(msg + "\n")
            await send_to_gc(msg)

            logger.info("gc_predefined_tasks> Execution completed.")
            # FLE-2167 Module Test Setup Changes
            # return "Completed"
            return(max_step_errors_x, max_step_errors_y)

    except Exception as ex:
        msg = f"{test_name} : {str(ex)}: {traceback.format_exc()}"
        logger.error(log_prefix + msg)
        await send_error_message_to_gc(msg, ex)
        GlobalVar.set_stop_gc(True)
        return

async def mover_stutter_test(mover = "pd", profile = 1, start_pos = 0, end_pos = 1, min_step = 0.1, iterations = 1):
    # 2023-02-27/kstruebing
    # Stresstest for single-axis-mover
    # Mover:
    # fm: FocusMover
    # fms: FilterModuleSlider
    # pd: PlateDoor
    # das1: DetectorApertureSlider1
    # das2: DetectorApertureSlider2
    # els: ExcitationLightSelector
    # bld: BottomLightDirector

    test_name = mover + '_' + sys._getframe().f_code.co_name    # get function name from sys
  
    distance = end_pos - start_pos
    count1 = int(distance / min_step)
    step_size = distance / count1

    msg = f'Starting {test_name}...'
    logger.info(log_prefix + msg)
    await send_to_gc(msg)

    t_mover = all_movers[mover]
    # FLE-2167 Module Test Setup Changes
    await mc6_unit.StartFirmware()
    await t_mover.InitializeDevice()

    msg = f'                    Mover: {mover}, profile: {profile} '
    #logger.info(msg)
    await send_to_gc(msg)
    msg = f'                 Distance: {distance} mm'
    #logger.info(msg)
    await send_to_gc(msg)
    msg = f'Nr of steps per direction: {count1}'
    #logger.info(msg)
    await send_to_gc(msg)
    msg = f'                Step_size: {step_size} mm'
    #logger.info(msg)
    await send_to_gc(msg)

    await t_mover.Home()
    await t_mover.UseProfile(profile)
    msg = f"      Initial step_errors: {t_mover.Mover.LastHomeStepErrors} msteps"
    #logger.info(msg)
    await send_to_gc(msg)

    cont = True
    idx = 0
    msg = f"Starting Test..."
    await send_to_gc(msg)

    try:
        GlobalVar.set_stop_gc(False)
        while idx < iterations and cont:
            if iterations > 1:
                msg = f'Iteration {idx + 1}'
                #logger.info(msg)
                await send_to_gc(msg)
            if not cont:
                await t_mover.Move(start_pos)

            new_pos = start_pos
            idx1 = 0

            while idx1 < count1 and cont:
                if GlobalVar.get_stop_gc():
                            msg = f"{test_name} : stopped by User"
                            logger.error(log_prefix + msg)
                            await send_to_gc(msg)
                            return(msg)
                new_pos = new_pos + 0.1
                if cont:
                    await t_mover.Move( new_pos)

                new_pos = new_pos + 0.9
                if cont:
                    await t_mover.Move( new_pos)

                new_pos = new_pos + (step_size - 1)
                if cont:
                    await t_mover.Move( new_pos)
                    
                idx1 = idx1 + 1

            idx1 = 0

            while idx1 < count1 and cont:
                if GlobalVar.get_stop_gc():
                            msg = f"{test_name} : stopped by User"
                            logger.error(log_prefix + msg)
                            await send_to_gc(msg)
                            return(msg)

                new_pos = new_pos - 0.1
                if cont:
                    await t_mover.Move(new_pos)

                new_pos = new_pos - 0.9
                if cont:
                    await t_mover.Move( new_pos)

                new_pos = new_pos - (step_size - 1)
                if cont:
                    await t_mover.Move( new_pos)
                    
                idx1 = idx1 + 1

            if cont:
                if GlobalVar.get_stop_gc():
                            msg = f"{test_name} : stopped by User"
                            logger.error(log_prefix + msg)
                            await send_to_gc(msg)
                            return(msg)

                await t_mover.Move( start_pos)
                msg = f'Running to start_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            if cont:
                if GlobalVar.get_stop_gc():
                            msg = f"{test_name} : stopped by User"
                            logger.error(log_prefix + msg)
                            await send_to_gc(msg)
                            return(msg)

                await t_mover.Move( end_pos)
                msg = f'Running to end_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            if cont:
                if GlobalVar.get_stop_gc():
                            msg = f"{test_name} : stopped by User"
                            logger.error(log_prefix + msg)
                            await send_to_gc(msg)
                            return(msg)

                await t_mover.Move( start_pos)
                msg = f'Running to start_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            idx = idx + 1

    except Exception as ex:

        msg = f"{test_name} -> {traceback.format_exc()}"
        logger.error(log_prefix + msg)
        await send_to_gc(msg, ex)

    await t_mover.Home()
    result = t_mover.Mover.LastHomeStepErrors
    return (result)

async def scantable_stutter_test(profile = 1, start_x_pos = 0, end_x_pos = 1,start_y_pos = 0, end_y_pos = 1, min_step = 0.1, iterations =1):
    # 2022-07-29/kstruebing (<- hderichs)
    # Derived from mover_stutter_test
    
    # Tested 2022-8-02/kstruebing on Newport: Y-Direction not properly working

    test_name = sys._getframe().f_code.co_name    # get function name from sys

    msg = f'Starting {test_name}...'
    #logger.info(log_prefix + msg)
    await send_to_gc(msg)
    
    distance_x = end_x_pos - start_x_pos
    distance_y = end_y_pos - start_y_pos
    #count1 = math.floor(distance / min_step)
    count1_x = int(distance_x / min_step)
    count1_y = int(distance_y / min_step)
    step_size_x = distance_x / count1_x
    step_size_y = distance_y / count1_y

    t_mover = all_movers["st"]

    msg = f'            Profile: {profile} '
    #logger.info(msg)
    await send_to_gc(msg)

    msg = f'         Distance X: {distance_x} mm'
    #logger.info(msg)
    await send_to_gc(msg)

    msg = f'         Distance Y: {distance_y} mm'
    #logger.info(msg)
    await send_to_gc(msg)

    msg = f'      Nr of steps X: {count1_x}'
    #logger.info(msg)
    await send_to_gc(msg)

    msg = f'      Nr of steps Y: {count1_y}'
    #logger.info(msg)
    await send_to_gc(msg)

    msg = f'        Step_size X: {step_size_x} mm'
    #logger.info(msg)
    await send_to_gc(msg)

    msg = f'        Step_size Y: {step_size_y} mm'
    #logger.info(msg)
    await send_to_gc(msg)

    step_errors = await t_mover.Home()
    await t_mover.UseProfile(profile)

    msg = f'Initial step_errors: {step_errors} msteps'
    #logger.info(msg)
    await send_to_gc(msg)
 
    cont = True
    idx = 0
    idy = 0

    try:
        # Iterate X-Positions
        while idx < iterations and cont:
            if iterations > 1:
                msg = f'Iteration {idx + 1}'
                #logger.info(msg)
                await send_to_gc(msg)

            if not cont:
                await t_mover.Move(start_x_pos, start_y_pos)

            new_x_pos = start_x_pos
            idx1 = 0

            while idx1 < count1_x and cont:
                new_x_pos = new_x_pos + 0.1
                if cont:
                    await t_mover.Move( new_x_pos, start_y_pos)

                new_x_pos = new_x_pos + 0.9
                if cont:
                    await t_mover.Move( new_x_pos, start_y_pos)

                new_x_pos = new_x_pos + (step_size_x - 1)
                if cont:
                    await t_mover.Move( new_x_pos, start_y_pos)
                idx1 = idx1 + 1

            idx1 = 0

            while idx1 < count1_x and cont:
                new_x_pos = new_x_pos - 0.1
                if cont:
                    await t_mover.Move( new_x_pos, start_y_pos)

                new_x_pos = new_x_pos - 0.9
                if cont:
                    await t_mover.Move( new_x_pos, start_y_pos)

                new_x_pos = new_x_pos - (step_size_x - 1)
                if cont:
                    await t_mover.Move( new_x_pos, start_y_pos)
                idx1 = idx1 + 1

            if cont:
                await t_mover.Move( start_x_pos, start_y_pos)
                msg = f'Running to start_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            if cont:
                await t_mover.Move( end_x_pos, end_y_pos)
                msg = f'Running to end_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            if cont:
                await t_mover.Move( start_x_pos, start_y_pos)
                msg = f'Running to start_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            idx = idx + 1

        # Iterate Y-Positions
        while idy < iterations and cont:
            if iterations > 1:
                msg = f'Iteration {idy + 1}'
                #logger.info(msg)
                await send_to_gc(msg)

            if not cont:
                await t_mover.Move(start_x_pos, start_y_pos)

            new_y_pos = start_y_pos
            idy1 = 0

            while idy1 < count1_y and cont:
                new_y_pos = new_y_pos + 0.1
                if cont:
                    await t_mover.Move( start_x_pos, new_y_pos)

                new_y_pos = new_y_pos + 0.9
                if cont:
                    await t_mover.Move( start_x_pos, new_y_pos)

                new_y_pos = new_y_pos + (step_size_y - 1)
                if cont:
                    await t_mover.Move( start_x_pos, new_y_pos)
                idy1 = idy1 + 1

            idy1 = 0

            while idy1 < count1_y and cont:
                new_y_pos = new_y_pos - 0.1
                if cont:
                    await t_mover.Move( start_x_pos, new_y_pos)

                new_y_pos = new_y_pos - 0.9
                if cont:
                    await t_mover.Move( start_x_pos, new_y_pos)

                new_y_pos = new_y_pos - (step_size_y - 1)
                if cont:
                    await t_mover.Move( new_x_pos, start_y_pos)
                idy1 = idy1 + 1

            if cont:
                await t_mover.Move( start_x_pos, start_y_pos)
                msg = f'Running to start_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            if cont:
                await t_mover.Move( end_x_pos, end_y_pos)
                msg = f'Running to end_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            if cont:
                await t_mover.Move( start_x_pos, start_y_pos)
                msg = f'Running to start_pos'
                #logger.info(msg)
                await send_to_gc(msg)

            idy = idy + 1


    except Exception as ex:

        msg = f"{error_msg}{test_name} : {str(ex)}: {traceback.format_exc()}"
        logger.error(msg)
        await send_to_gc(msg)

    result = await t_mover.Home()
    msg = f"{complete_msg}{test_name}: step_errors: {result} msteps"
    #logger.info(msg)
    await send_to_gc(msg)
    return (result)

async def mover_limit(mv = "fms", pre_home_offset = 2.0, start = 0, stop = 179, step = 0.1, threshold = 200):
    # Find mechanical limit of  mover
    # 
    # ==> Max Position of Mover has to be changed before running this test !
    # 
    # Move from start until mechanical blockade or stop position
    # Moves step per step and checks step errors
    # If step errors exceed threshold, the actual position will be returned
    # Mover:
    # fm: FocusMover
    # fms: FilterModuleSlider
    # pd: PlateDoor
    # das1: DetectorApertureSlider1
    # das2: DetectorApertureSlider2
    # els: ExcitationLightSelector: Producing high step errors end_pos > 355 (>5000msteps)
    # bld: BottomLightDirector: Producing high step errors end_pos > 345 (>5000msteps)

    # Tested 2022-08-02/kstruebing with Filtermodule Slider on Newport

    test_name = sys._getframe().f_code.co_name    # get function name from sys

    msg = f'Starting {test_name}...'
    logger.info(log_prefix + msg)
    await send_to_gc(msg)

    t_mover = all_movers[mv] 

    await t_mover.Home()

    position = start

    GlobalVar.set_stop_gc(False)

    while position < stop:

        if GlobalVar.get_stop_gc():
            msg = f"{stop_msg}{test_name} : Stop Button pressed by User"
            logger.error(msg)
            await send_to_gc(msg)
            return 

        await t_mover.UseProfile(1)
        await t_mover.Move(position)
        msg = f'{info_msg}{test_name} : Mover Position: {await t_mover.GetPosition()}, Step Errors: {t_mover.Mover.LastHomeStepErrors}'
        await send_to_gc(msg)
        
        await t_mover.Move(pre_home_offset)
        await t_mover.Home()
        if abs(t_mover.Mover.LastHomeStepErrors) > threshold: 
            msg = f'{info_msg}{test_name} : Mover Limit Position: {await t_mover.GetPosition()}, Step Errors: {t_mover.Mover.LastHomeStepErrors}'
            await send_to_gc(msg)
            return(position)
            
        position += step
    
    msg = f'{complete_msg}{test_name} : Mover: {mv}: No limit between {start} and {stop}'
    await send_to_gc(msg)
    return(msg)

async def mover_performance_test(mover = 'pd', profile = 1, cycles = 1):
    # [x] Terminal Test 2024-01-29 (Newport)
    # [x] GC Test 2024-01-29 (Newport)
    # Last modified: 2024-01-29/kstruebing
    # Initial

    # Test of speed limits with fast profile
    # profile: mover profile: the fastest !
    # pd = 25mm/s
    # fm = 5mm/s
    # fms = 150mm/s
    # els = 189°/s
    # bld = 189°/s
    # das1/2 = 7mm/s

    test_name = sys._getframe().f_code.co_name    # get function name from sys

    msg = f'Starting {test_name}...'
    logger.info(log_prefix + msg)
    await send_to_gc(msg)

    t_mover =all_movers[mover]

    await t_mover.Home()
    await t_mover.UseProfile(profile)
    start_position = t_mover.get_config(pd_enum.Positions.Min)
    stop_position = t_mover.get_config(pd_enum.Positions.Max)

    duration_s = 0.0
    for i in range(cycles):
        await t_mover.Move(start_position)
        start_time = time.perf_counter()
        await t_mover.Move(stop_position)
        stop_time = time.perf_counter()
        duration_s += stop_time - start_time
    await t_mover.Home()
    speed = (stop_position - start_position)/(duration_s/cycles)
    msg = f'{test_name} Result: {speed:.3f}mm/s'
    return(msg)
